#ifndef HEADER_LUMITRONIX_IFLEX_WORKSHOP_BUTTON
#define HEADER_LUMITRONIX_IFLEX_WORKSHOP_BUTTON


#include <Arduino.h>


/**
 * Preconfigured active low push down buttons for use with 1700076-E11270 during
 * the iFlex Workshop.
 */
static constexpr auto LUMITRONIX_IFLEX_WORKSHOP_BUTTON_A_PIN{12};
static constexpr auto LUMITRONIX_IFLEX_WORKSHOP_BUTTON_B_PIN{11};
static constexpr decltype(::millis()) LUMITRONIX_IFLEX_WORKSHOP_BUTTON_DEBOUNCE{50};


enum class LUMITRONIX_iFlexWorkshop_ButtonState {
	BUTTON_PRESSED,
	BUTTON_RELEASED,
};


class LUMITRONIX_iFlexWorkshop_ActiveLowButton {
private:
	uint8_t pin;
	LUMITRONIX_iFlexWorkshop_ButtonState stableState{LUMITRONIX_iFlexWorkshop_ButtonState::BUTTON_RELEASED};
	LUMITRONIX_iFlexWorkshop_ButtonState lastState{LUMITRONIX_iFlexWorkshop_ButtonState::BUTTON_RELEASED};
	decltype(::millis()) lastStateChange{0};

public:
	explicit LUMITRONIX_iFlexWorkshop_ActiveLowButton(uint8_t pin) : pin{pin} {
	}

	void Begin() {
		::pinMode(pin, INPUT_PULLUP);
	}

	[[nodiscard]] bool IsPressed() {
		Update();
		return LUMITRONIX_iFlexWorkshop_ButtonState::BUTTON_PRESSED == stableState;
	}

	[[nodiscard]] bool IsReleased() {
		Update();
		return LUMITRONIX_iFlexWorkshop_ButtonState::BUTTON_RELEASED == stableState;
	}

private:
	void Update() {
		LUMITRONIX_iFlexWorkshop_ButtonState const currentState = Read();
		decltype(::millis()) const now = ::millis();

		if (currentState == lastState) {
			if ((now - lastStateChange) > LUMITRONIX_IFLEX_WORKSHOP_BUTTON_DEBOUNCE) {
				stableState = lastState;
			}
			return;
		}

		lastState = currentState;
		lastStateChange = now;
	}

	[[nodiscard]] LUMITRONIX_iFlexWorkshop_ButtonState Read() {
		if (LOW == ::digitalRead(pin)) {
			return LUMITRONIX_iFlexWorkshop_ButtonState::BUTTON_PRESSED;
		} else {
			return LUMITRONIX_iFlexWorkshop_ButtonState::BUTTON_RELEASED;
		}
	}
};


class LUMITRONIX_iFlexWorkshop_ButtonA final : public LUMITRONIX_iFlexWorkshop_ActiveLowButton {
public:
	explicit LUMITRONIX_iFlexWorkshop_ButtonA() : LUMITRONIX_iFlexWorkshop_ActiveLowButton{LUMITRONIX_IFLEX_WORKSHOP_BUTTON_A_PIN} {
	}
};


class LUMITRONIX_iFlexWorkshop_ButtonB final : public LUMITRONIX_iFlexWorkshop_ActiveLowButton {
public:
	explicit LUMITRONIX_iFlexWorkshop_ButtonB() : LUMITRONIX_iFlexWorkshop_ActiveLowButton{LUMITRONIX_IFLEX_WORKSHOP_BUTTON_B_PIN} {
	}
};


#endif

